/// <reference path="/njres/js/comlib/neo.js.src/000neo.js" />

///
/// Ajax処理
///
/// 追加オプション
///   detectMultiError - multicmdの個々のJSONのエラーを解析する(規定値はfalse)
/// 
/// ・RESTエラーになった場合にステータスコードがRESTのエラー番号になる。
/// ・(jQuery 1.5以上)RESTエラーになった場合にstatus=successでエラーイベントが呼ばれる。
/// ・$.ajaxSetupで次のパラメーターが無視される:type, dataType, global, beforeSend, success, error, complete
neo.ajax = {
    NEOAJAX_OBJJSON: 'json',
    NEOAJAX_OBJXML: 'xml',

    defaultOptions: { // Ajaxオプションの初期値
        "type": "POST",
        "dataType": "xml"
    },

    _connections: {}, // 現在の接続
    _reserved: {}, // 接続予約
    _numRequests: 0, // リクエスト数

    rest: function (options) {
        /// <summary>$.ajax()と同じようにNEOJAPAN内部形式のRESTを解析する。</summary>
        /// <param name="options" type="Object">neo.ajaxのオプション</param>
        var id = options['id'];
        // 同じIDで通信中なら後で
        if (!!id && !!this._connections[id]) {
            this._addToReserved(id, options);
            return;
        }

        this._execute(options);
    },

    submitForm: function ($form, options) {
        /// <summary>フォームの内容を送信し、NEOJAPAN内部形式のRESTを解析する</summary>
        /// <param name="$form" type="jQuery">フォーム</param>
        /// <param name="options" type="Object">ajaxFormのoptions</param>
        var sendOptions = this._getOptionsToSend(options);
        $form.ajaxForm(sendOptions);
    },

    abort: function (id, stopAll) {
        /// <summary>通信を中止する</summary>
        /// <param name="id" type="String">rest()で指定したoptions.id</param>
        /// <param name="stopAll" type="Boolean">接続待ちも中止するならtrue</param>
        if (stopAll !== false && !!this._reserved[id]) {
            delete this._reserved[id];
        }
        if (!!this._connections[id]) {
            this._connections[id].abort();
        }
    },

    getDataValue: function (xhr, key) {
        /// <summary>XML、又はJSON(multi可)から指定したキーの値を取得する</summary>
        /// <param name="xhr" type="XMLHttpRequest">XML HTTP Request</param>
        /// <param name="key" type="String">要素名(XML)、プロパティ名(JSON)</param>
        if (!!xhr.responseXML) {
            var $node = $(xhr.responseXML).find(key);
            if ($node.length > 0) {
                return $node.text();
            }
        } else {
            try {
                return this._getJsonDataValue($.parseJSON(xhr.responseText), key);
            } catch (errorInfo) { }
        }
    },

    getErrorMessage: function (xhr, error) {
        /// <summary>エラーメッセージを取得する</summary>
        /// <param name="xhr" type="XMLHttpRequest">XML HTTP Request</param>
        /// <param name="error" type="String">jQueryのエラー文字列</param>
        /// <returns type="String" />
        var code = !!xhr ? parseInt(xhr.status) : 200;
        var status = '', message = '';
        if (code == 0) { // 接続エラー
            status = String(xhr.statusText);
            message = neo.Resource.ajax[status + "_errorMessage"];
        } else if (code >= 400) { // HTTPエラー
            status = String(xhr.status);
            message = neo.Resource.ajax[status + "_errorMessage"];
        } else { // 独自エラー(HTTP通信は成功)
            message = error;
        }

        if (typeof (message) == "string") {
            return message;
        }
        return neo.Resource.ajax["errorMessage"].replace("{{status}}", status);
    },

    //- Private(通信処理) ------------------------------------------------------
    _execute: function (options) {
        /// <summary>通信を実行する</summary>
        /// <param name="options" type="Object">neo.ajaxのオプション</param>
        var id = options['id'];
        var xhr = $.ajax(this._getOptionsToSend(options));
        if (!!id) { // 1つ目の接続
            this._connections[id] = xhr;
        }
    },

    _getOptionsToSend: function (options) {
        /// <summary>送信用オプションを取得する。</summary>
        /// <param name="options" type="Object">追加options</param>
        /// <returns type="Object" />
        var self = this;
        var neoOptions = {};

        options.global = typeof (options.global) == "undefined" || !!(options.global);
        $.extend(neoOptions, this.defaultOptions, options);

        var forcedOptions = {
            "global": false,
            "beforeSend": function (xhr, jqOptions) {
                self._onBeforeSend(xhr, jqOptions, neoOptions);
            },
            "success": function (data, status, xhr) {
                self._parse(data, status, xhr, neoOptions);
            },
            "error": function (xhr, status, error) {
                self._onError(xhr, status, error, neoOptions);
            },
            "complete": function (xhr, status) {
                self._onComplete(xhr, status, neoOptions);
            }
        };
        return $.extend({}, neoOptions, forcedOptions);
    },

    //- Private(レスポンス処理) ------------------------------------------------
    _parse: function (data, status, xhr, options) {
        /// <summary>XML RESTを解析する。</summary>
        /// <param name="data" type="Object">レスポンス</param>
        /// <param name="status" type="String">ステータス</param>
        /// <param name="xhr" type="XMLHttpRequest">XML Http Request</param>
        /// <param name="options" type="Object">neo.ajaxのオプション</param>
        if (options.dataType == 'json') {
            this._parseJson(data, status, xhr, options);
        } else if (options.dataType == 'html') {
            this._parseHtml(data, status, xhr, options);
        } else {
            this._parseXml(data, status, xhr, options);
        }
    },

    _parseXml: function (xml, status, xhr, options) {
        /// <summary>XML RESTを解析する。</summary>
        /// <param name="xml" type="DOM">XML</param>
        /// <param name="status" type="String">ステータス</param>
        /// <param name="xhr" type="XMLHttpRequest">XML Http Request</param>
        /// <param name="options" type="Object">neo.ajaxのオプション</param>
        var $xml = $(xml.documentElement);
        if ($xml.is("parsererror")) {
            options._dataStatus = 'parsererror';
            this._onError(xhr, 'parsererror', $xml.find("parsererror").text(), options);
        } else if ($xml.find("status").eq(0).text() == "ok") { // BTMANTIS0015248 V20R00
            this._onSuccess(xml, status, xhr, options);
        } else {
            var errorNo = $xml.find("errorno").eq(0).text(); // BTMANTIS0015248 V20R00
            if ($.isFunction(options.errorconvmsg)) {
                var message = options.errorconvmsg(this.NEOAJAX_OBJXML, $xml);
            }
            else {
                message = $xml.find("errormessage").eq(0).text(); // BTMANTIS0015248 V20R00
            }
            this._parseErrorData(errorNo, message, status, xhr, options);
        }
    },

    _parseJson: function (json, status, xhr, options) {
        /// <summary>JSON RESTを解析する。</summary>
        /// <param name="json" type="Object">JSON</param>
        /// <param name="status" type="String">ステータス</param>
        /// <param name="xhr" type="XMLHttpRequest">XML Http Request</param>
        /// <param name="options" type="Object">neo.ajaxのオプション</param>
        var error = this._getSingleJsonError(json, options);
        if (!!error) {
            this._parseErrorData(error.errorno, error.message, status, xhr, options);
            return;
        }

        if (!!options.detectMultiError) {
            for (var i = 0; !!json[i.toString()]; i++) {
                var error = this._getSingleJsonError(json[i.toString()], options);
                if (!!error) {
                    this._parseErrorData(error.errorno, error.message, status, xhr, options);
                    return;
                }
            }
        }

        this._onSuccess(json, status, xhr, options);
    },

    _parseHtml: function (cHtml, status, xhr, options) {
        /// <summary>HTMLレスポンスを解析する。</summary>
        /// <param name="cHtml" type="String">HTML文字列</param>
        /// <param name="status" type="String">ステータス</param>
        /// <param name="xhr" type="XMLHttpRequest">XML Http Request</param>
        /// <param name="options" type="Object">neo.ajaxのオプション</param>
        this._onSuccess(cHtml, status, xhr, options);
    },

    _parseErrorData: function (errorNo, errorMessage, status, xhr, options) {
        /// <summary>エラー情報を解析する。</summary>
        /// <param name="errorNo" type="String">エラー番号</param>
        /// <param name="errorMessage" type="String">エラーメッセージ</param>
        /// <param name="status" type="String">ステータス</param>
        /// <param name="xhr" type="XMLHttpRequest">XML Http Request</param>
        /// <param name="options" type="Object">neo.ajaxのオプション</param>
        var number = parseInt(errorNo);
        var message = (typeof (errorMessage) == "string") ? errorMessage : "";
        if ($.fn.jquery >= "1.5" && number < 0) {
            xhr.status = number;
        }
        this._onError(xhr, status, message, options);
    },

    _getSingleJsonError: function (json, options) {
        /// <summary>JSON REST(multiを除く)のエラー情報を取得する。</summary>
        /// <param name="json" type="Object">JSON</param>
        /// <param name="options" type="Object">neo.ajaxのオプション</param>
        if (json == null) {
            return { 'errorno': 0, 'message': '' };
        } else if (json.status == 'ok') {
            return null;
        } else {
            var errorNo = json.errorno;
            var message = json.errormessage;
            if ($.isFunction(options.errorconvmsg)) {
                var message = options.errorconvmsg(this.NEOAJAX_OBJJSON, json);
            }
            return { 'errorno': errorNo, 'message': message };
        }
    },

    //- Private(全体通信管理) --------------------------------------------------
    _onBeforeRequest: function (options) {
        /// <summary>リクエスト前処理を行う。</summary>
        /// <param name="options" type="Object">neo.ajaxのオプション</param>
        if (options.global && this._numRequests <= 0) {
            $(document).trigger("ajaxStart");
        }
        this._numRequests++;
    },

    _onAfterRequest: function (options) {
        /// <summary>リクエスト後処理を行う</summary>
        /// <param name="options" type="Object">neo.ajaxのオプション</param>
        this._numRequests--;
        if (options.global && this._numRequests <= 0) {
            $(document).trigger("ajaxStop");
        }
        if (!!options['id']) {
            this._startNextReserved(options['id']);
        }
    },

    _addToReserved: function (id, options) {
        /// <summary>接続予約に追加する</summary>
        /// <param name="id" type="String">接続ID</param>
        /// <param name="options" type="Object">neo.ajaxのオプション</param>
        if (!this._reserved[id]) {
            this._reserved[id] = [options];
        } else {
            this._reserved[id].push(options);
        }
    },

    _startNextReserved: function (id) {
        /// <summary>次の接続予約通信を開始する</summary>
        /// <param name="id" type="String">接続ID</param>
        var self = this;
        if (!!self._connections[id]) {
            delete self._connections[id];
        }
        if (!!self._reserved[id]) {
            var nextOpts = self._reserved[id].shift();
            if (self._reserved[id].length <= 0) {
                delete self._reserved[id];
            }
            setTimeout(function () { self._execute(nextOpts); }, 0);
        }
    },

    //- Private(個別通信管理) --------------------------------------------------
    _onBeforeSend: function (xhr, jqOptions, options) {
        /// <summary>送信前処理を行う。</summary>
        /// <param name="xhr" type="XMLHttpRequest">XML Http Request</param>
        /// <param name="jqOptions" type="Object">jQueryのオプション</param>
        /// <param name="options" type="Object">neo.ajaxのオプション</param>
        this._onBeforeRequest(options);
        // 個別の通信前処理を実行
        if ($.isFunction(options.beforeSend)) {
            options.beforeSend(xhr, jqOptions);
        }
        // 共通の通信前処理を実行
        if (options.global) {
            $(document).trigger("ajaxSend", [xhr, jqOptions]);
        }
    },

    _onSuccess: function (data, status, xhr, options) {
        /// <summary>成功時処理を行う。</summary>
        /// <param name="data" type="Object">レスポンス</param>
        /// <param name="status" type="String">ステータス</param>
        /// <param name="xhr" type="XMLHttpRequest">XML Http Request</param>
        /// <param name="options" type="Object">neo.ajaxのオプション</param>
        // 個別の通信成功処理を実行
        if ($.isFunction(options.success)) {
            options.success(data, status, xhr);
        }
        // 共通の通信成功処理を実行
        if (options.global) {
            $(document).trigger("ajaxSuccess", [xhr, options]);
        }
    },

    _onError: function (xhr, status, error, options) {
        /// <summary>失敗時処理を行う。</summary>
        /// <param name="xhr" type="XMLHttpRequest">XML Http Request</param>
        /// <param name="status" type="String">ステータス</param>
        /// <param name="error" type="String">エラーメッセージ</param>
        /// <param name="options" type="Object">neo.ajaxのオプション</param>
        var message = this.getErrorMessage(xhr, error);
        // 個別の通信失敗処理を実行
        if ($.isFunction(options.error)) {
            options.error(xhr, status, message);
        }
        // 共通の通信失敗処理を実行
        if (options.global) {
            $("body").trigger("ajaxError", [xhr, options, message]);
        }
    },

    _onComplete: function (xhr, status, options) {
        /// <summary>通信後処理を行う。</summary>
        /// <param name="xhr" type="XMLHttpRequest">XML Http Request</param>
        /// <param name="status" type="String">ステータス</param>
        /// <param name="options" type="Object">neo.ajaxのオプション</param>
        if (options._dataStatus) {
            status = options._dataStatus;
        }
        // 個別の通信終了処理を実行
        if ($.isFunction(options.complete)) {
            options.complete(xhr, status);
        }
        // 共通の通信終了処理を実行
        if (options.global) {
            $(document).trigger("ajaxComplete", [xhr, options]);
        }
        this._onAfterRequest(options);
    },

    //- Private(値の編集) ------------------------------------------------------
    _getJsonDataValue: function (json, key) {
        /// <summary>JSON(multi可)から指定したキーの値を取得する</summary>
        /// <param name="json" type="Object">JSON</param>
        /// <param name="key" type="String">プロパティ名</param>
        var i = 0, jsonItem = json;
        while (typeof jsonItem == 'object') {
            var value = jsonItem[key]
            if (typeof value != 'undefined') {
                return value;
            }

            jsonItem = json[i.toString()];
            i++;
        }
    }
};
